<?php

/*
 * This file is part of the API Platform project.
 *
 * (c) Kévin Dunglas <dunglas@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace ApiPlatform\Serializer\Mapping\Loader;

use ApiPlatform\Metadata\ApiProperty;
use ApiPlatform\Metadata\Property\Factory\PropertyNameCollectionFactoryInterface;
use Illuminate\Database\Eloquent\Model;
use Symfony\Component\Serializer\Attribute\Context;
use Symfony\Component\Serializer\Attribute\DiscriminatorMap;
use Symfony\Component\Serializer\Attribute\Groups;
use Symfony\Component\Serializer\Attribute\Ignore;
use Symfony\Component\Serializer\Attribute\MaxDepth;
use Symfony\Component\Serializer\Attribute\SerializedName;
use Symfony\Component\Serializer\Attribute\SerializedPath;
use Symfony\Component\Serializer\Mapping\AttributeMetadata;
use Symfony\Component\Serializer\Mapping\AttributeMetadataInterface;
use Symfony\Component\Serializer\Mapping\ClassDiscriminatorMapping;
use Symfony\Component\Serializer\Mapping\ClassMetadataInterface;
use Symfony\Component\Serializer\Mapping\Loader\LoaderInterface;
use Symfony\Component\Serializer\NameConverter\NameConverterInterface;

/**
 * Loader for PHP attributes using ApiProperty.
 */
final class PropertyMetadataLoader implements LoaderInterface
{
    public function __construct(
        private readonly PropertyNameCollectionFactoryInterface $propertyNameCollectionFactory,
        private readonly ?NameConverterInterface $nameConverter = null,
    ) {
    }

    public function loadClassMetadata(ClassMetadataInterface $classMetadata): bool
    {
        // It's very weird to grab Eloquent's properties in that case as they're never serialized
        // the Serializer makes a call on the abstract class, let's save some unneeded work with a condition
        if (Model::class === $classMetadata->getName()) {
            return false;
        }

        $refl = $classMetadata->getReflectionClass();
        $attributes = [];
        $classGroups = [];
        $classContextAnnotation = null;

        foreach ($refl->getAttributes(ApiProperty::class) as $clAttr) {
            $this->addAttributeMetadata($clAttr->newInstance(), $attributes);
        }

        $attributesMetadata = $classMetadata->getAttributesMetadata();

        foreach ($refl->getAttributes() as $a) {
            $attribute = $a->newInstance();
            if ($attribute instanceof DiscriminatorMap) {
                $classMetadata->setClassDiscriminatorMapping(new ClassDiscriminatorMapping(
                    method_exists($attribute, 'getTypeProperty') ? $attribute->getTypeProperty() : $attribute->typeProperty,
                    method_exists($attribute, 'getMapping') ? $attribute->getMapping() : $attribute->mapping
                ));
                continue;
            }

            if ($attribute instanceof Groups) {
                $classGroups = method_exists($attribute, 'getGroups') ? $attribute->getGroups() : $attribute->groups;

                continue;
            }

            if ($attribute instanceof Context) {
                $classContextAnnotation = $attribute;
            }
        }

        foreach ($refl->getProperties() as $reflProperty) {
            foreach ($reflProperty->getAttributes(ApiProperty::class) as $propAttr) {
                $this->addAttributeMetadata($propAttr->newInstance()->withProperty($reflProperty->name), $attributes);
            }
        }

        foreach ($refl->getMethods() as $reflMethod) {
            foreach ($reflMethod->getAttributes(ApiProperty::class) as $methodAttr) {
                $this->addAttributeMetadata($methodAttr->newInstance()->withProperty($reflMethod->getName()), $attributes);
            }
        }

        foreach ($this->propertyNameCollectionFactory->create($classMetadata->getName()) as $propertyName) {
            if (!isset($attributesMetadata[$propertyName])) {
                $attributesMetadata[$propertyName] = new AttributeMetadata($propertyName);
                $classMetadata->addAttributeMetadata($attributesMetadata[$propertyName]);
            }

            foreach ($classGroups as $group) {
                $attributesMetadata[$propertyName]->addGroup($group);
            }

            if ($classContextAnnotation) {
                $this->setAttributeContextsForGroups($classContextAnnotation, $attributesMetadata[$propertyName]);
            }

            if (!isset($attributes[$propertyName])) {
                continue;
            }

            $attributeMetadata = $attributesMetadata[$propertyName];

            // This code is adapted from Symfony\Component\Serializer\Mapping\Loader\AttributeLoader
            foreach ($attributes[$propertyName] as $attr) {
                if ($attr instanceof Groups) {
                    $groups = method_exists($attr, 'getGroups') ? $attr->getGroups() : $attr->groups;
                    foreach ($groups as $group) {
                        $attributeMetadata->addGroup($group);
                    }
                    continue;
                }

                match (true) {
                    $attr instanceof MaxDepth => $attributeMetadata->setMaxDepth(method_exists($attr, 'getMaxDepth') ? $attr->getMaxDepth() : $attr->maxDepth),
                    $attr instanceof SerializedName => $attributeMetadata->setSerializedName(method_exists($attr, 'getSerializedName') ? $attr->getSerializedName() : $attr->serializedName),
                    $attr instanceof SerializedPath => $attributeMetadata->setSerializedPath(method_exists($attr, 'getSerializedPath') ? $attr->getSerializedPath() : $attr->serializedPath),
                    $attr instanceof Ignore => $attributeMetadata->setIgnore(true),
                    $attr instanceof Context => $this->setAttributeContextsForGroups($attr, $attributeMetadata),
                    default => null,
                };
            }
        }

        return true;
    }

    /**
     * @param array<string, array<mixed>> $attributes
     */
    private function addAttributeMetadata(ApiProperty $attribute, array &$attributes): void
    {
        if (($prop = $this->nameConverter?->denormalize($attribute->getProperty()) ?? $attribute->getProperty()) && ($value = $attribute->getSerialize())) {
            $attributes[$prop] = $value;
        }
    }

    private function setAttributeContextsForGroups(Context $annotation, AttributeMetadataInterface $attributeMetadata): void
    {
        $context = method_exists($annotation, 'getContext') ? $annotation->getContext() : $annotation->context;
        $groups = method_exists($annotation, 'getGroups') ? $annotation->getGroups() : $annotation->groups;
        $normalizationContext = method_exists($annotation, 'getNormalizationContext') ? $annotation->getNormalizationContext() : $annotation->normalizationContext;
        $denormalizationContext = method_exists($annotation, 'getDenormalizationContext') ? $annotation->getDenormalizationContext() : $annotation->denormalizationContext;

        if ($normalizationContext || $context) {
            $attributeMetadata->setNormalizationContextForGroups($normalizationContext ?: $context, $groups);
        }

        if ($denormalizationContext || $context) {
            $attributeMetadata->setDenormalizationContextForGroups($denormalizationContext ?: $context, $groups);
        }
    }
}
