<?php

namespace App\State;

use ApiPlatform\Metadata\Operation;
use ApiPlatform\State\ProcessorInterface;
use App\Entity\User;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Email;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Uid\Uuid;

class UserPasswordHasher implements ProcessorInterface
{
    public function __construct(
        private ProcessorInterface $processor,
        private UserPasswordHasherInterface $passwordHasher,
        private MailerInterface $mailer,
        private string $appUrl
    ) {}

    public function process($data, Operation $operation, array $uriVariables = [], array $context = [])
    {
        if (!$data instanceof User) {
            return $this->processor->process($data, $operation, $uriVariables, $context);
        }

        // Hash password if plainPassword is provided
        if ($data->getPlainPassword()) {
            $hashedPassword = $this->passwordHasher->hashPassword(
                $data,
                $data->getPlainPassword()
            );
            $data->setPassword($hashedPassword);
            $data->eraseCredentials();
        }

        // For new users (registration), generate verification token
        if (!$data->getId()) {
            $data->setVerificationToken(Uuid::v4()->toRfc4122());

            // Persist the user first
            $result = $this->processor->process($data, $operation, $uriVariables, $context);

            // Send verification email after user is saved
            $this->sendVerificationEmail($data);

            return $result;
        }

        // For existing users (updates)
        return $this->processor->process($data, $operation, $uriVariables, $context);
    }

    private function sendVerificationEmail(User $user): void
    {
        // Use frontend URL for verification page
        $verificationUrl = sprintf(
            '%s/api/verify-email?token=%s',
            rtrim($this->appUrl, '/'),
            $user->getVerificationToken()
        );

        $email = (new Email())
            ->from('info@auwebx.com')
            ->to($user->getEmail())
            ->subject('Please verify your email')
            ->html($this->getEmailTemplate($user->getFirstName(), $verificationUrl));

        try {
            $this->mailer->send($email);
        } catch (\Exception $e) {
            // Log error but don't break the registration process
            // You might want to use a logger service here
        }
    }

    private function getEmailTemplate(string $firstName, string $verificationUrl): string
    {
        return sprintf('
            <!DOCTYPE html>
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                    .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                    .button {
                        display: inline-block;
                        padding: 12px 24px;
                        background-color: #28a745;
                        color: #ffffff;
                        text-decoration: none;
                        border-radius: 4px;
                        margin: 20px 0;
                    }
                    .footer {
                        margin-top: 30px;
                        padding-top: 20px;
                        border-top: 1px solid #ddd;
                        font-size: 12px;
                        color: #666;
                    }
                    .token-box {
                        background-color: #f5f5f5;
                        padding: 15px;
                        border-radius: 5px;
                        margin: 20px 0;
                        word-break: break-all;
                        font-family: monospace;
                        font-size: 14px;
                    }
                </style>
            </head>
            <body>
                <div class="container">
                    <h2>Welcome! Please verify your email</h2>
                    <p>Hello %s,</p>
                    <p>Thank you for registering! Please verify your email address by clicking the button below:</p>
                    <a href="%s" class="button">Verify Email Address</a>
                    <p><strong>Note:</strong> This link will take you to our verification page where the process will complete automatically.</p>
                    <div class="footer">
                        <p>If you did not create an account, please ignore this email.</p>
                        <p>This verification link is valid for 24 hours.</p>
                    </div>
                </div>
            </body>
            </html>
        ', $firstName, $verificationUrl);
    }
}
