<?php

namespace App\EventSubscriber;

use ApiPlatform\Symfony\EventListener\EventPriorities;
use App\Entity\PracticeTest;
use App\Repository\QuestionRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ViewEvent;
use Symfony\Component\HttpKernel\KernelEvents;

final class PracticeTestSubscriber implements EventSubscriberInterface
{
    private const QUESTIONS_PER_TEST = 50; // Change this number as needed

    public function __construct(
        private readonly EntityManagerInterface $em,
        private readonly QuestionRepository $questionRepository
    ) {
    }

    public static function getSubscribedEvents(): array
    {
        return [
            KernelEvents::VIEW => ['populateQuestions', EventPriorities::PRE_WRITE],
        ];
    }

    public function populateQuestions(ViewEvent $event): void
    {
        $practiceTest = $event->getControllerResult();
        $request = $event->getRequest();

        // Only handle POST requests for PracticeTest creation
        if (
            !$practiceTest instanceof PracticeTest ||
            $request->getMethod() !== 'POST' ||
            $request->attributes->get('_api_resource_class') !== PracticeTest::class
        ) {
            return;
        }

        $exam = $practiceTest->getExam();
        if (!$exam) {
            return; // Should not happen due to validation, but safe guard
        }

        // Get all questions for this exam
        $allQuestions = $this->questionRepository->createQueryBuilder('q')
            ->where('q.exam = :exam')
            ->setParameter('exam', $exam)
            ->getQuery()
            ->getResult();

        if (empty($allQuestions)) {
            // Optional: you could throw an exception or add zero questions
            return;
        }

        // Group questions by subject
        $questionsBySubject = [];
        foreach ($allQuestions as $question) {
            $subject = $question->getSubject();
            $subjectId = $subject?->getId() ?? 'no-subject';
            $questionsBySubject[$subjectId][] = $question;
        }

        $selectedQuestions = [];

        // Calculate how many questions per subject (proportional)
        $totalQuestions = count($allQuestions);
        foreach ($questionsBySubject as $subjectId => $questions) {
            $subjectCount = count($questions);
            $ratio = $subjectCount / $totalQuestions;
            $targetCount = (int) round($ratio * self::QUESTIONS_PER_TEST);

            // Ensure at least 1 question per subject if possible
            $targetCount = max(1, min($targetCount, $subjectCount));

            // Shuffle and take
            shuffle($questions);
            $selectedQuestions = array_merge($selectedQuestions, array_slice($questions, 0, $targetCount));
        }

        // If we have fewer than desired due to rounding, fill up randomly
        if (count($selectedQuestions) < self::QUESTIONS_PER_TEST) {
            $remaining = self::QUESTIONS_PER_TEST - count($selectedQuestions);
            $remainingQuestions = array_filter(
                $allQuestions,
                fn($q) => !in_array($q, $selectedQuestions, true)
            );
            shuffle($remainingQuestions);
            $selectedQuestions = array_merge($selectedQuestions, array_slice($remainingQuestions, 0, $remaining));
        }

        // If somehow too many (edge case), trim
        if (count($selectedQuestions) > self::QUESTIONS_PER_TEST) {
            $selectedQuestions = array_slice($selectedQuestions, 0, self::QUESTIONS_PER_TEST);
        }

        // Add selected questions to the practice test
        foreach ($selectedQuestions as $question) {
            $practiceTest->addQuestion($question);
        }

        // Optional: persist immediately if needed (not required with PRE_WRITE)
        // $this->em->persist($practiceTest);
    }
}
