<?php

namespace App\Entity;

use ApiPlatform\Metadata\ApiResource;
use ApiPlatform\Metadata\ApiProperty;
use ApiPlatform\Metadata\Get;
use ApiPlatform\Metadata\GetCollection;
use ApiPlatform\Metadata\Post;
use ApiPlatform\Metadata\Put;
use ApiPlatform\Metadata\Patch;
use App\Repository\SubscriptionRepository;
use App\State\SubscriptionProcessor;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Serializer\Attribute\Groups;
use Symfony\Component\Validator\Constraints as Assert;

#[ORM\Entity(repositoryClass: SubscriptionRepository::class)]
#[ORM\Table(name: 'subscriptions')]
#[ApiResource(
    operations: [
        new GetCollection(
            security: "is_granted('ROLE_ADMIN') or is_granted('ROLE_USER')",
            normalizationContext: ['groups' => ['subscription:read'], 'enable_max_depth' => true]
        ),
        new Get(
            security: "is_granted('ROLE_ADMIN') or (is_granted('ROLE_USER') and object.getUser() == user)",
            normalizationContext: ['groups' => ['subscription:read'], 'enable_max_depth' => true]
        ),
        new Post(
            security: "is_granted('ROLE_USER')",
            denormalizationContext: ['groups' => ['subscription:create']],
            validationContext: ['groups' => ['Default']],
            processor: SubscriptionProcessor::class
        ),
        new Patch(
            security: "is_granted('ROLE_ADMIN')",
            denormalizationContext: ['groups' => ['subscription:admin_update']],
            normalizationContext: ['groups' => ['subscription:read']]
        ),
        new Put(
            security: "is_granted('ROLE_ADMIN')",
            denormalizationContext: ['groups' => ['subscription:admin_update']],
            normalizationContext: ['groups' => ['subscription:read']]
        )
    ],
    normalizationContext: ['groups' => ['subscription:read'], 'enable_max_depth' => true],
    denormalizationContext: ['groups' => ['subscription:write']]
)]
class Subscription
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    #[Groups(['subscription:read'])]
    private ?int $id = null;

    #[ORM\Column]
    #[Assert\NotBlank(message: "Amount is required")]
    #[Assert\GreaterThanOrEqual(value: 5000, message: "Minimum subscription amount is 5000")]
    #[Groups(['subscription:read', 'subscription:create'])]
    private ?float $amount = null;

    #[ORM\Column(length: 50)]
    #[Assert\NotBlank(message: "Payment method is required")]
    #[Assert\Choice(choices: ['paystack', 'bank_transfer'], message: "Payment method must be either 'paystack' or 'bank_transfer'")]
    #[Groups(['subscription:read', 'subscription:create'])]
    private ?string $paymentMethod = null;

    #[ORM\Column(length: 50)]
    #[Groups(['subscription:read', 'subscription:admin_update'])]
    private string $status = 'pending';

    #[ORM\Column(length: 255, nullable: true)]
    #[Groups(['subscription:read', 'subscription:create'])]
    private ?string $paystackReference = null;

    #[ORM\Column(length: 255, nullable: true)]
    #[Groups(['subscription:read', 'subscription:create'])]
    private ?string $transactionReference = null;

    #[ORM\Column(type: 'text', nullable: true)]
    #[Groups(['subscription:read', 'subscription:admin_update'])]
    private ?string $adminNotes = null;

    #[ORM\Column]
    #[Groups(['subscription:read'])]
    private ?\DateTimeImmutable $createdAt = null;

    #[ORM\Column(nullable: true)]
    #[Groups(['subscription:read'])]
    private ?\DateTimeImmutable $confirmedAt = null;

    #[ORM\ManyToOne(targetEntity: User::class)]
    #[ORM\JoinColumn(nullable: false, onDelete: "CASCADE")]
    #[Assert\NotNull(
        message: "User is required",
        groups: ['subscription:internal']
    )]
    #[Groups(['subscription:read'])]
    #[ApiProperty(readableLink: true)]
    private ?User $user = null;

    public function __construct()
    {
        $this->createdAt = new \DateTimeImmutable();
    }

    // Getters and Setters

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getAmount(): ?float
    {
        return $this->amount;
    }

    public function setAmount(float $amount): self
    {
        $this->amount = $amount;
        return $this;
    }

    public function getPaymentMethod(): ?string
    {
        return $this->paymentMethod;
    }

    public function setPaymentMethod(string $paymentMethod): self
    {
        $this->paymentMethod = $paymentMethod;
        return $this;
    }

    public function getStatus(): string
    {
        return $this->status;
    }

    public function setStatus(string $status): self
    {
        $this->status = $status;
        return $this;
    }

    public function getPaystackReference(): ?string
    {
        return $this->paystackReference;
    }

    public function setPaystackReference(?string $paystackReference): self
    {
        $this->paystackReference = $paystackReference;
        return $this;
    }

    public function getTransactionReference(): ?string
    {
        return $this->transactionReference;
    }

    public function setTransactionReference(?string $transactionReference): self
    {
        $this->transactionReference = $transactionReference;
        return $this;
    }

    public function getAdminNotes(): ?string
    {
        return $this->adminNotes;
    }

    public function setAdminNotes(?string $adminNotes): self
    {
        $this->adminNotes = $adminNotes;
        return $this;
    }

    public function getCreatedAt(): ?\DateTimeImmutable
    {
        return $this->createdAt;
    }

    public function setCreatedAt(\DateTimeImmutable $createdAt): self
    {
        $this->createdAt = $createdAt;
        return $this;
    }

    public function getConfirmedAt(): ?\DateTimeImmutable
    {
        return $this->confirmedAt;
    }

    public function setConfirmedAt(?\DateTimeImmutable $confirmedAt): self
    {
        $this->confirmedAt = $confirmedAt;
        return $this;
    }

    public function getUser(): ?User
    {
        return $this->user;
    }

    public function setUser(?User $user): self
    {
        $this->user = $user;
        return $this;
    }
}
