<?php

namespace App\Entity;

use ApiPlatform\Metadata\ApiResource;
use ApiPlatform\Metadata\ApiProperty;
use ApiPlatform\Metadata\Get;
use ApiPlatform\Metadata\GetCollection;
use ApiPlatform\Metadata\Post;
use ApiPlatform\Metadata\Put;
use ApiPlatform\Metadata\Delete;
use App\Repository\QuestionRepository;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Serializer\Attribute\Groups;
use Symfony\Component\Validator\Constraints as Assert;

#[ORM\Entity(repositoryClass: QuestionRepository::class)]
#[ORM\Table(name: 'questions')]
#[ApiResource(
    operations: [
        new GetCollection(security: "is_granted('ROLE_ADMIN') or is_granted('ROLE_STAFF') or is_granted('ROLE_USER')"),
        new Get(security: "is_granted('ROLE_ADMIN') or is_granted('ROLE_STAFF') or is_granted('ROLE_USER')"),
        new Post(security: "is_granted('ROLE_ADMIN') or is_granted('ROLE_STAFF')"),
        new Put(security: "is_granted('ROLE_ADMIN') or is_granted('ROLE_STAFF')"),
        new Delete(security: "is_granted('ROLE_ADMIN') or is_granted('ROLE_STAFF')")
    ],
    normalizationContext: ['groups' => ['question:read']],
    denormalizationContext: ['groups' => ['question:write']]
)]
class Question
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    #[Groups(['question:read'])]
    private ?int $id = null;

    #[ORM\ManyToOne(targetEntity: Exam::class)]
    #[ORM\JoinColumn(nullable: false, onDelete: "CASCADE")]
    #[Assert\NotNull(message: "Exam is required")]
    #[Groups(['question:read', 'question:write'])]
    #[ApiProperty(readableLink: true)]
    private ?Exam $exam = null;

    #[ORM\ManyToOne(targetEntity: Subject::class)]
    #[ORM\JoinColumn(nullable: false, onDelete: "CASCADE")]
    #[Assert\NotNull(message: "Subject is required")]
    #[Groups(['question:read', 'question:write'])]
    #[ApiProperty(readableLink: true)]
    private ?Subject $subject = null;

    #[ORM\ManyToOne(targetEntity: Topic::class)]
    #[ORM\JoinColumn(nullable: false, onDelete: "CASCADE")]
    #[Assert\NotNull(message: "Topic is required")]
    #[Groups(['question:read', 'question:write'])]
    #[ApiProperty(readableLink: true)]
    private ?Topic $topic = null;

    #[ORM\Column(type: 'text')]
    #[Assert\NotBlank(message: "Question text is required")]
    #[Groups(['question:read', 'question:write'])]
    private ?string $questionText = null;

    #[ORM\Column(type: 'json')]
    #[Assert\NotBlank(message: "Options are required")]
    #[Assert\Count(min: 2, minMessage: "At least 2 options are required")]
    #[Groups(['question:read', 'question:write'])]
    private array $options = [];

    #[ORM\Column(length: 10)]
    #[Assert\NotBlank(message: "Correct option is required")]
    #[Groups(['question:read', 'question:write'])]
    private ?string $correctOption = null;

    #[ORM\Column(type: 'text', nullable: true)]
    #[Groups(['question:read', 'question:write'])]
    private ?string $solution = null;

    #[ORM\Column(length: 20)]
    #[Assert\NotBlank(message: "Difficulty is required")]
    #[Assert\Choice(choices: ['easy', 'medium', 'hard'], message: "Difficulty must be easy, medium, or hard")]
    #[Groups(['question:read', 'question:write'])]
    private ?string $difficulty = null;

    #[ORM\Column(length: 20)]
    #[Assert\NotBlank(message: "Source is required")]
    #[Assert\Choice(choices: ['past', 'ai', 'user'], message: "Source must be past, ai, or user")]
    #[Groups(['question:read', 'question:write'])]
    private ?string $source = null;

    #[ORM\Column(type: 'integer', nullable: true)]
    #[Assert\Range(min: 1900, max: 2100, notInRangeMessage: "Year must be between {{ min }} and {{ max }}")]
    #[Groups(['question:read', 'question:write'])]
    private ?int $year = null;

    // Getters and Setters

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getExam(): ?Exam
    {
        return $this->exam;
    }

    public function setExam(?Exam $exam): self
    {
        $this->exam = $exam;
        return $this;
    }

    public function getSubject(): ?Subject
    {
        return $this->subject;
    }

    public function setSubject(?Subject $subject): self
    {
        $this->subject = $subject;
        return $this;
    }

    public function getTopic(): ?Topic
    {
        return $this->topic;
    }

    public function setTopic(?Topic $topic): self
    {
        $this->topic = $topic;
        return $this;
    }

    public function getQuestionText(): ?string
    {
        return $this->questionText;
    }

    public function setQuestionText(string $questionText): self
    {
        $this->questionText = $questionText;
        return $this;
    }

    public function getOptions(): array
    {
        return $this->options;
    }

    public function setOptions(array $options): self
    {
        $this->options = $options;
        return $this;
    }

    public function getCorrectOption(): ?string
    {
        return $this->correctOption;
    }

    public function setCorrectOption(string $correctOption): self
    {
        $this->correctOption = $correctOption;
        return $this;
    }

    public function getSolution(): ?string
    {
        return $this->solution;
    }

    public function setSolution(?string $solution): self
    {
        $this->solution = $solution;
        return $this;
    }

    public function getDifficulty(): ?string
    {
        return $this->difficulty;
    }

    public function setDifficulty(string $difficulty): self
    {
        $this->difficulty = $difficulty;
        return $this;
    }

    public function getSource(): ?string
    {
        return $this->source;
    }

    public function setSource(string $source): self
    {
        $this->source = $source;
        return $this;
    }

    public function getYear(): ?int
    {
        return $this->year;
    }

    public function setYear(?int $year): self
    {
        $this->year = $year;
        return $this;
    }


    #[Groups(['question:read'])]
public function getOptionA(): ?string
{
    return $this->options['A'] ?? null;
}

#[Groups(['question:read'])]
public function getOptionB(): ?string
{
    return $this->options['B'] ?? null;
}

#[Groups(['question:read'])]
public function getOptionC(): ?string
{
    return $this->options['C'] ?? null;
}

#[Groups(['question:read'])]
public function getOptionD(): ?string
{
    return $this->options['D'] ?? null;
}
}
