<?php

namespace App\Controller;

use App\Entity\Transaction;
use App\Repository\SubscriptionRepository;
use App\Repository\TransactionRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Bundle\SecurityBundle\Security;

final class UserFinanceController extends AbstractController
{
    #[Route('/api/me/finance', name: 'user_finance', methods: ['GET'])]
    public function getFinance(
        SubscriptionRepository $subscriptionRepo,
        TransactionRepository $transactionRepo,
        Security $security
    ): JsonResponse {
        $user = $security->getUser();

        if (!$user) {
            return $this->json(['error' => 'Unauthorized'], 401);
        }

        // Calculate total verified payments (credits)
        $totalCredits = $subscriptionRepo->createQueryBuilder('s')
            ->select('SUM(s.amount)')
            ->where('s.user = :user')
            ->andWhere('s.status = :status')
            ->setParameter('user', $user)
            ->setParameter('status', 'verified')
            ->getQuery()
            ->getSingleScalarResult() ?? 0;

        // Calculate total debits (deductions)
        $totalDebits = $transactionRepo->createQueryBuilder('t')
            ->select('SUM(t.amount)')
            ->where('t.user = :user')
            ->andWhere('t.type = :type')
            ->setParameter('user', $user)
            ->setParameter('type', 'debit')
            ->getQuery()
            ->getSingleScalarResult() ?? 0;

        // Current balance = credits - debits
        $balance = $totalCredits - $totalDebits;

        $pending = $subscriptionRepo->count([
            'user' => $user,
            'status' => 'pending'
        ]);

        $verifiedCount = $subscriptionRepo->count([
            'user' => $user,
            'status' => 'verified'
        ]);

        return $this->json([
            'balance' => (float) $balance,
            'totalPaid' => (float) $totalCredits,
            'totalSpent' => (float) $totalDebits,
            'verifiedCount' => $verifiedCount,
            'pendingCount' => $pending
        ]);
    }

    #[Route('/api/me/deduct-balance', name: 'user_deduct_balance', methods: ['POST'])]
    public function deductBalance(
        Request $request,
        SubscriptionRepository $subscriptionRepo,
        TransactionRepository $transactionRepo,
        EntityManagerInterface $em,
        Security $security
    ): JsonResponse {
        $user = $security->getUser();

        if (!$user) {
            return $this->json(['error' => 'Unauthorized'], 401);
        }

        // Get request data
        $data = json_decode($request->getContent(), true);

        if (!isset($data['amount']) || !is_numeric($data['amount'])) {
            return $this->json(['error' => 'Invalid amount'], 400);
        }

        $amount = (float) $data['amount'];
        $description = $data['description'] ?? 'Deduction';

        if ($amount <= 0) {
            return $this->json(['error' => 'Amount must be greater than 0'], 400);
        }

        // Calculate current balance
        $totalCredits = $subscriptionRepo->createQueryBuilder('s')
            ->select('SUM(s.amount)')
            ->where('s.user = :user')
            ->andWhere('s.status = :status')
            ->setParameter('user', $user)
            ->setParameter('status', 'verified')
            ->getQuery()
            ->getSingleScalarResult() ?? 0;

        $totalDebits = $transactionRepo->createQueryBuilder('t')
            ->select('SUM(t.amount)')
            ->where('t.user = :user')
            ->andWhere('t.type = :type')
            ->setParameter('user', $user)
            ->setParameter('type', 'debit')
            ->getQuery()
            ->getSingleScalarResult() ?? 0;

        $currentBalance = $totalCredits - $totalDebits;

        // Check if sufficient balance
        if ($currentBalance < $amount) {
            return $this->json([
                'error' => 'Insufficient balance',
                'message' => 'You do not have enough balance to complete this action',
                'required' => $amount,
                'available' => $currentBalance,
                'shortfall' => $amount - $currentBalance
            ], 400);
        }

        // Create transaction record
        $transaction = new Transaction();
        $transaction->setUser($user);
        $transaction->setAmount($amount);
        $transaction->setType('debit');
        $transaction->setDescription($description);
        $transaction->setBalanceAfter($currentBalance - $amount);

        $em->persist($transaction);
        $em->flush();

        return $this->json([
            'success' => true,
            'message' => 'Balance deducted successfully',
            'deducted' => $amount,
            'balance' => $currentBalance - $amount,
            'transaction' => [
                'id' => $transaction->getId(),
                'amount' => $transaction->getAmount(),
                'description' => $transaction->getDescription(),
                'createdAt' => $transaction->getCreatedAt()->format('Y-m-d H:i:s')
            ]
        ]);
    }

    #[Route('/api/me/transactions', name: 'user_transactions', methods: ['GET'])]
    public function getTransactions(
        TransactionRepository $transactionRepo,
        Security $security
    ): JsonResponse {
        $user = $security->getUser();

        if (!$user) {
            return $this->json(['error' => 'Unauthorized'], 401);
        }

        $transactions = $transactionRepo->getUserTransactions($user->getId());

        $data = array_map(function ($transaction) {
            return [
                'id' => $transaction->getId(),
                'amount' => $transaction->getAmount(),
                'type' => $transaction->getType(),
                'description' => $transaction->getDescription(),
                'balanceAfter' => $transaction->getBalanceAfter(),
                'createdAt' => $transaction->getCreatedAt()->format('Y-m-d H:i:s')
            ];
        }, $transactions);

        return $this->json([
            'transactions' => $data,
            'total' => count($data)
        ]);
    }
}
