<?php

namespace App\Controller;

use App\Entity\Subscription;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/api')]
class SubscriptionController extends AbstractController
{
    #[Route('/subscriptions/{id}/confirm', name: 'subscription_confirm', methods: ['PATCH'])]
    #[IsGranted('ROLE_ADMIN')]
    public function confirmSubscription(
        int $id,
        Request $request,
        EntityManagerInterface $entityManager
    ): JsonResponse {
        $subscription = $entityManager->getRepository(Subscription::class)->find($id);

        if (!$subscription) {
            return $this->json(['error' => 'Subscription not found'], 404);
        }

        if ($subscription->getStatus() === 'confirmed') {
            return $this->json(['error' => 'Subscription already confirmed'], 400);
        }

        $data = json_decode($request->getContent(), true);

        // Update subscription status
        $subscription->setStatus('confirmed');
        $subscription->setConfirmedAt(new \DateTimeImmutable());

        if (isset($data['adminNotes'])) {
            $subscription->setAdminNotes($data['adminNotes']);
        }

        $entityManager->flush();

        // TODO: Send confirmation email to user
        // TODO: Activate user's subscription/premium features

        return $this->json([
            'message' => 'Subscription confirmed successfully',
            'subscription' => [
                'id' => $subscription->getId(),
                'status' => $subscription->getStatus(),
                'confirmedAt' => $subscription->getConfirmedAt()?->format('Y-m-d H:i:s'),
            ]
        ]);
    }

    #[Route('/subscriptions/{id}/reject', name: 'subscription_reject', methods: ['PATCH'])]
    #[IsGranted('ROLE_ADMIN')]
    public function rejectSubscription(
        int $id,
        Request $request,
        EntityManagerInterface $entityManager
    ): JsonResponse {
        $subscription = $entityManager->getRepository(Subscription::class)->find($id);

        if (!$subscription) {
            return $this->json(['error' => 'Subscription not found'], 404);
        }

        if ($subscription->getStatus() === 'confirmed') {
            return $this->json(['error' => 'Cannot reject confirmed subscription'], 400);
        }

        $data = json_decode($request->getContent(), true);

        // Update subscription status
        $subscription->setStatus('rejected');

        if (isset($data['adminNotes'])) {
            $subscription->setAdminNotes($data['adminNotes']);
        }

        $entityManager->flush();

        // TODO: Send rejection notification to user

        return $this->json([
            'message' => 'Subscription rejected',
            'subscription' => [
                'id' => $subscription->getId(),
                'status' => $subscription->getStatus(),
            ]
        ]);
    }

    #[Route('/subscriptions/pending', name: 'subscriptions_pending', methods: ['GET'])]
    #[IsGranted('ROLE_ADMIN')]
    public function getPendingSubscriptions(EntityManagerInterface $entityManager): JsonResponse
    {
        $subscriptions = $entityManager->getRepository(Subscription::class)
            ->findBy(['status' => 'pending'], ['createdAt' => 'DESC']);

        $result = array_map(function ($sub) {
            return [
                'id' => $sub->getId(),
                'amount' => $sub->getAmount(),
                'paymentMethod' => $sub->getPaymentMethod(),
                'transactionReference' => $sub->getTransactionReference(),
                'createdAt' => $sub->getCreatedAt()?->format('Y-m-d H:i:s'),
                'user' => [
                    'id' => $sub->getUser()?->getId(),
                    'email' => $sub->getUser()?->getEmail(),
                ]
            ];
        }, $subscriptions);

        return $this->json($result);
    }
}
