<?php

namespace App\Controller;

use App\Entity\User;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Attribute\AsController;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Email;
use Symfony\Component\Uid\Uuid;

#[AsController]
class ResendVerificationController extends AbstractController
{
    public function __construct(
        private EntityManagerInterface $em,
        private MailerInterface $mailer,
        private string $appUrl
    ) {}

    public function __invoke(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $email = $data['email'] ?? null;

        if (!$email) {
            return new JsonResponse(['error' => 'Email is required'], 400);
        }

        $user = $this->em->getRepository(User::class)->findOneBy(['email' => $email]);

        if (!$user) {
            // Don't reveal if email exists
            return new JsonResponse([
                'message' => 'If the email exists and is not verified, a new verification link has been sent.'
            ], 200);
        }

        if ($user->isVerified()) {
            return new JsonResponse(['message' => 'Email is already verified'], 200);
        }

        // Generate new verification token
        $user->setVerificationToken(Uuid::v4()->toRfc4122());
        $this->em->flush();

        // Send verification email
        $this->sendVerificationEmail($user);

        return new JsonResponse([
            'message' => 'If the email exists and is not verified, a new verification link has been sent.'
        ], 200);
    }

    private function sendVerificationEmail(User $user): void
    {
        $verificationUrl = sprintf(
            '%s/verify-email?token=%s',
            rtrim($this->appUrl, '/'),
            $user->getVerificationToken()
        );

        $email = (new Email())
            ->from('info@auwebx.com')
            ->to($user->getEmail())
            ->subject('Verify your email - New link')
            ->html(sprintf('
                <!DOCTYPE html>
                <html>
                <head>
                    <style>
                        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                        .button {
                            display: inline-block;
                            padding: 12px 24px;
                            background-color: #28a745;
                            color: #ffffff;
                            text-decoration: none;
                            border-radius: 4px;
                            margin: 20px 0;
                        }
                    </style>
                </head>
                <body>
                    <div class="container">
                        <h2>Verify Your Email</h2>
                        <p>Hello %s,</p>
                        <p>You requested a new verification link. Click the button below to verify your email:</p>
                        <a href="%s" class="button">Verify Email Address</a>
                        <p>This link is valid for 24 hours.</p>
                    </div>
                </body>
                </html>
            ', $user->getFirstName(), $verificationUrl));

        try {
            $this->mailer->send($email);
        } catch (\Exception $e) {
            // Log error
        }
    }
}
