<?php

namespace App\Controller;

use App\Entity\User;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Log\LoggerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Attribute\AsController;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Email;
use Symfony\Component\Uid\Uuid;

#[AsController]
class ForgotPasswordController extends AbstractController
{
    public function __construct(
        private EntityManagerInterface $em,
        private MailerInterface $mailer,
        private LoggerInterface $logger,
        private string $appUrl,
        private string $frontendUrl  // Add this parameter
    ) {}

    public function __invoke(Request $request): JsonResponse
    {
        $data = json_decode($request->getContent(), true);

        $email = $data['email'] ?? null;

        if (!$email || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return new JsonResponse(['message' => 'A valid email is required'], 400);
        }

        $user = $this->em->getRepository(User::class)->findOneBy(['email' => $email]);

        // Always return success message for security
        if (!$user) {
            return new JsonResponse([
                'message' => 'If the email exists in our system, a password reset link has been sent.'
            ], 200);
        }

        $resetToken = Uuid::v4()->toRfc4122();
        $user->setResetToken($resetToken);
        $user->setResetTokenExpiresAt(new \DateTimeImmutable('+1 hour'));
        $this->em->flush();

        // CHANGED: Point to frontend instead of backend API
        $resetUrl = sprintf('%s/reset-password?token=%s', $this->frontendUrl, $resetToken);

        $emailObj = (new Email())
            ->from('info@auwebx.com')
            ->to($user->getEmail())
            ->subject('Password Reset Request')
            ->html($this->getEmailTemplate($user->getFirstName(), $resetUrl));

        try {
            $this->mailer->send($emailObj);
        } catch (\Exception $e) {
            $this->logger->error('Failed to send password reset email: ' . $e->getMessage());
        }

        return new JsonResponse([
            'message' => 'If the email exists in our system, a password reset link has been sent.'
        ], 200);
    }

    private function getEmailTemplate(string $firstName, string $resetUrl): string
    {
        $template = <<<HTML
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Password Reset</title>
</head>
<body style="margin: 0; padding: 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif; background-color: #f4f7fa;">
    <table width="100%" cellpadding="0" cellspacing="0" style="background-color: #f4f7fa; padding: 40px 0;">
        <tr>
            <td align="center">
                <table width="600" cellpadding="0" cellspacing="0" style="background-color: #ffffff; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.05);">
                    <!-- Header -->
                    <tr>
                        <td style="padding: 40px 40px 30px; text-align: center; background: linear-gradient(135deg, #16a34a 0%, #15803d 100%); border-radius: 8px 8px 0 0;">
                            <h1 style="margin: 0; color: #ffffff; font-size: 28px; font-weight: 600;">Password Reset</h1>
                        </td>
                    </tr>

                    <!-- Content -->
                    <tr>
                        <td style="padding: 40px;">
                            <p style="margin: 0 0 20px; color: #333333; font-size: 16px; line-height: 1.6;">
                                Hello <strong>{{FIRST_NAME}}</strong>,
                            </p>

                            <p style="margin: 0 0 30px; color: #666666; font-size: 15px; line-height: 1.6;">
                                We received a request to reset your password for your Secure Online Examination Platform account. Click the button below to create a new password:
                            </p>

                            <!-- Button -->
                            <table width="100%" cellpadding="0" cellspacing="0">
                                <tr>
                                    <td align="center" style="padding: 20px 0;">
                                        <a href="{{RESET_URL}}" style="display: inline-block; padding: 16px 40px; background: linear-gradient(135deg, #16a34a 0%, #15803d 100%); color: #ffffff; text-decoration: none; border-radius: 6px; font-size: 16px; font-weight: 600; box-shadow: 0 4px 12px rgba(22, 163, 74, 0.4);">
                                            Reset Password
                                        </a>
                                    </td>
                                </tr>
                            </table>

                            <p style="margin: 30px 0 20px; color: #999999; font-size: 14px; line-height: 1.6;">
                                Or copy and paste this link into your browser:
                            </p>

                            <p style="margin: 0 0 30px; padding: 12px; background-color: #f3f4f6; border-radius: 4px; color: #666666; font-size: 13px; word-break: break-all;">
                                {{RESET_URL}}
                            </p>

                            <p style="margin: 0; color: #999999; font-size: 14px; line-height: 1.6;">
                                If you didn't request this password reset, you can safely ignore this email. The link will expire in 1 hour for security reasons.
                            </p>
                        </td>
                    </tr>

                    <!-- Footer -->
                    <tr>
                        <td style="padding: 30px 40px; background-color: #f8f9fa; border-radius: 0 0 8px 8px; text-align: center;">
                            <p style="margin: 0 0 10px; color: #999999; font-size: 13px;">
                                © 2025 Secure Online Examination Platform. All rights reserved.
                            </p>
                            <p style="margin: 0; color: #cccccc; font-size: 12px;">
                                This is an automated message, please do not reply.
                            </p>
                        </td>
                    </tr>
                </table>
            </td>
        </tr>
    </table>
</body>
</html>
HTML;

        return str_replace(
            ['{{FIRST_NAME}}', '{{RESET_URL}}'],
            [$firstName, $resetUrl],
            $template
        );
    }
}

/* curl -X POST http://localhost:8000/api/forgot-password \
  -H "Content-Type: application/json" \
  -d '{"email":"auwebx87@gmail.com"}' */
